<?php
/**
 * Run Scheduled Notifications (Cron Worker)
 *
 * This endpoint should be triggered by a cron job or scheduler.
 * It reads due scheduled_notifications from Firestore and sends them via FCM.
 */

header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/fcm_helper.php';

$projectId = 'arqana-bd662';
$serviceAccountFile = __DIR__ . '/arqana-bd662-firebase-adminsdk-fbsvc-d8399e5e41.json';

// Optional simple protection: require a secret key via query string (?key=...)
// Set your own secret value here if you want basic protection.
$CRON_SECRET = null; // e.g. 'my-very-long-secret';

if ($CRON_SECRET !== null) {
    $provided = $_GET['key'] ?? '';
    if (!hash_equals($CRON_SECRET, $provided)) {
        http_response_code(403);
        echo json_encode([
            'success' => false,
            'error' => 'Forbidden'
        ]);
        exit();
    }
}

try {
    // Build Firestore runQuery request to find due pending notifications
    $nowUtc = (new DateTime('now', new DateTimeZone('UTC')))->format(DateTime::ATOM);

    $structuredQuery = [
        'structuredQuery' => [
            'from' => [
                ['collectionId' => 'scheduled_notifications'],
            ],
            'where' => [
                'compositeFilter' => [
                    'op' => 'AND',
                    'filters' => [
                        [
                            'fieldFilter' => [
                                'field' => ['fieldPath' => 'status'],
                                'op' => 'EQUAL',
                                'value' => ['stringValue' => 'pending'],
                            ],
                        ],
                        [
                            'fieldFilter' => [
                                'field' => ['fieldPath' => 'scheduled_at'],
                                'op' => 'LESS_THAN_OR_EQUAL',
                                'value' => ['timestampValue' => $nowUtc],
                            ],
                        ],
                    ],
                ],
            ],
        ],
    ];

    $accessToken = getFirestoreAccessToken($serviceAccountFile);

    $queryUrl = sprintf(
        'https://firestore.googleapis.com/v1/projects/%s/databases/(default)/documents:runQuery',
        $projectId
    );

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $queryUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json; charset=utf-8',
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($structuredQuery));

    $result = curl_exec($ch);

    if ($result === false) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('cURL error: ' . $error);
    }

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode < 200 || $httpCode >= 300) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to query Firestore for scheduled notifications',
            'firestore_response' => json_decode($result, true),
        ]);
        exit();
    }

    $rows = json_decode($result, true);

    $totalDue = 0;
    $sent = 0;
    $failed = 0;
    $details = [];

    if (is_array($rows)) {
        foreach ($rows as $row) {
            if (!isset($row['document'])) {
                continue;
            }

            $doc = $row['document'];
            $name = $doc['name'] ?? null;
            $fields = $doc['fields'] ?? [];

            $title = $fields['title']['stringValue'] ?? null;
            $message = $fields['message']['stringValue'] ?? null;
            $topic = $fields['topic']['stringValue'] ?? null;

            if (!$name || !$title || !$message || !$topic) {
                $failed++;
                $details[] = [
                    'document' => $name,
                    'status' => 'skipped_invalid_fields',
                ];
                continue;
            }

            $totalDue++;

            try {
                $sendResult = sendFCMv1($title, $message, $topic);

                if ($sendResult['http_code'] === 200) {
                    $sent++;
                    $details[] = [
                        'document' => $name,
                        'status' => 'sent',
                    ];

                    // Mark as sent in Firestore (status=sent, sent_at=now)
                    $updateUrl = sprintf(
                        'https://firestore.googleapis.com/v1/%s?updateMask.fieldPaths=status&updateMask.fieldPaths=sent_at',
                        $name // must not be urlencoded; Firestore expects full resource path with slashes
                    );

                    $updateBody = [
                        'fields' => [
                            'status' => ['stringValue' => 'sent'],
                            'sent_at' => ['timestampValue' => $nowUtc],
                        ],
                    ];

                    $chUpdate = curl_init();
                    curl_setopt($chUpdate, CURLOPT_URL, $updateUrl);
                    curl_setopt($chUpdate, CURLOPT_CUSTOMREQUEST, 'PATCH');
                    curl_setopt($chUpdate, CURLOPT_HTTPHEADER, [
                        'Authorization: Bearer ' . $accessToken,
                        'Content-Type: application/json; charset=utf-8',
                    ]);
                    curl_setopt($chUpdate, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($chUpdate, CURLOPT_POSTFIELDS, json_encode($updateBody));
                    curl_exec($chUpdate);
                    curl_close($chUpdate);
                } else {
                    $failed++;
                    $details[] = [
                        'document' => $name,
                        'status' => 'fcm_failed',
                        'fcm_http_code' => $sendResult['http_code'],
                        'fcm_response' => $sendResult['response'],
                    ];

                    // Optionally mark as failed in Firestore so it doesn't stay pending forever
                    $updateUrl = sprintf(
                        'https://firestore.googleapis.com/v1/%s?updateMask.fieldPaths=status',
                        $name
                    );
                    $updateBody = [
                        'fields' => [
                            'status' => ['stringValue' => 'failed'],
                        ],
                    ];
                    $chUpdate = curl_init();
                    curl_setopt($chUpdate, CURLOPT_URL, $updateUrl);
                    curl_setopt($chUpdate, CURLOPT_CUSTOMREQUEST, 'PATCH');
                    curl_setopt($chUpdate, CURLOPT_HTTPHEADER, [
                        'Authorization: Bearer ' . $accessToken,
                        'Content-Type: application/json; charset=utf-8',
                    ]);
                    curl_setopt($chUpdate, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($chUpdate, CURLOPT_POSTFIELDS, json_encode($updateBody));
                    curl_exec($chUpdate);
                    curl_close($chUpdate);
                }
            } catch (Throwable $e) {
                $failed++;
                $details[] = [
                    'document' => $name,
                    'status' => 'error',
                    'error' => $e->getMessage(),
                ];
            }
        }
    }

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Scheduled notifications processed',
        'total_due' => $totalDue,
        'sent' => $sent,
        'failed' => $failed,
        'details' => $details,
    ]);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage(),
    ]);
}


