<?php
/**
 * FCM Notification API Endpoint
 * Accepts POST requests to send push notifications via Firebase Cloud Messaging
 */

// Set CORS headers for mobile app access
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Method not allowed. Only POST requests are accepted.'
    ]);
    exit();
}

// Include the FCM helper
require __DIR__ . '/fcm_helper.php';

/**
 * Parse request body - supports both JSON and form data
 */
function parseRequestBody() {
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    
    // Check if JSON
    if (strpos($contentType, 'application/json') !== false) {
        $rawBody = file_get_contents('php://input');
        $data = json_decode($rawBody, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return null;
        }
        
        return $data;
    }
    
    // Otherwise, assume form data
    return $_POST;
}

/**
 * Validate required parameters
 */
function validateParameters($data) {
    $required = ['title', 'message', 'topic'];
    $missing = [];
    
    foreach ($required as $field) {
        if (!isset($data[$field]) || trim($data[$field]) === '') {
            $missing[] = $field;
        }
    }
    
    return $missing;
}

// Parse request body
$requestData = parseRequestBody();

if ($requestData === null) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Invalid JSON in request body.'
    ]);
    exit();
}

// Validate required parameters
$missingFields = validateParameters($requestData);

if (!empty($missingFields)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Missing required fields: ' . implode(', ', $missingFields)
    ]);
    exit();
}

// Extract parameters
$title = trim($requestData['title']);
$message = trim($requestData['message']);
$topic = trim($requestData['topic']);

// Send notification
try {
    $result = sendFCMv1($title, $message, $topic);

    // Check if the request was successful (HTTP 200)
    if ($result['http_code'] === 200) {
        $responseData = json_decode($result['response'], true);

        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Notification sent successfully',
            'data' => $responseData
        ]);
    } else {
        // FCM API returned an error
        $errorData = json_decode($result['response'], true);
        $errorMessage = $errorData['error']['message'] ?? 'Failed to send notification';

        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $errorMessage,
            'fcm_response' => $errorData
        ]);
    }
} catch (\Throwable $e) {
    // Handle any error/exception and always return JSON
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage(),
    ]);
}

