<?php
/**
 * PHP Video Upload Handler
 * Handles video uploads with dynamic directory structure
 */

// Set content type to JSON
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Method not allowed. Only POST requests are accepted.'
    ]);
    exit;
}

// Check if both video and path are provided
if (!isset($_FILES['video']) || !isset($_POST['path'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Missing required fields. Both video and path are required.'
    ]);
    exit;
}

$uploadedFile = $_FILES['video'];
$pathString = $_POST['path'];
$domain = "https://vendor.arqana.shop/";

// Validate file upload
if ($uploadedFile['error'] !== UPLOAD_ERR_OK) {
    $errorMessages = [
        UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize directive in php.ini',
        UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE directive in HTML form',
        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
    ];

    $errorMessage = $errorMessages[$uploadedFile['error']] ?? 'Unknown upload error';

    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $errorMessage
    ]);
    exit;
}

// Basic file size validation (50MB max)
$maxFileSize = 50 * 1024 * 1024; // 50MB in bytes
if ($uploadedFile['size'] > $maxFileSize) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'File size exceeds maximum allowed size of 50MB.'
    ]);
    exit;
}

// Validate file is a video
$allowedMimeTypes = [
    'video/mp4',
    'video/webm',
    'video/quicktime'
];

$fileMimeType = mime_content_type($uploadedFile['tmp_name']);
if (!in_array($fileMimeType, $allowedMimeTypes)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Invalid file type. Only video files (MP4, WebM, MOV) are allowed.'
    ]);
    exit;
}

// Sanitize path string to prevent directory traversal attacks
$sanitizedPath = preg_replace('/[^a-zA-Z0-9\/_-]/', '', $pathString);
$sanitizedPath = trim($sanitizedPath, '/');

// Remove any remaining dangerous patterns
$sanitizedPath = str_replace(['..', '//', '\\'], '', $sanitizedPath);

if (empty($sanitizedPath)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Invalid path string. Path must contain only alphanumeric characters, slashes, hyphens, and underscores.'
    ]);
    exit;
}

// Create target directory
$targetDir = 'uploads/' . $sanitizedPath;

// Create directory if it doesn't exist
if (!is_dir($targetDir)) {
    if (!mkdir($targetDir, 0755, true)) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to create directory structure.'
        ]);
        exit;
    }
}

// Get original file extension
$originalName = $uploadedFile['name'];
$fileExtension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));

// Validate extension matches MIME type
$extensionMimeMap = [
    'mp4' => ['video/mp4'],
    'webm' => ['video/webm'],
    'mov' => ['video/quicktime']
];

if (!isset($extensionMimeMap[$fileExtension]) || !in_array($fileMimeType, $extensionMimeMap[$fileExtension])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'File extension does not match file type.'
    ]);
    exit;
}

// Generate unique filename (timestamp + random string)
$uniqueFilename = time() . '_' . bin2hex(random_bytes(8)) . '.' . $fileExtension;
$targetPath = $targetDir . '/' . $uniqueFilename;

// Move uploaded file to target directory
if (!move_uploaded_file($uploadedFile['tmp_name'], $targetPath)) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to move uploaded file to destination.'
    ]);
    exit;
}

// Return success response
http_response_code(200);
echo json_encode([
    'success' => true,
    'message' => 'Video uploaded successfully.',
    'file_path' => $domain . $targetPath,
    'filename' => $uniqueFilename
]);
?>